/*jshint esversion: 6 */

define([
	"immutable", "lodash", "src/utils",
	"lib/tasks/treeOfMatrix", "lib/tasks/listOfTransform", "lib/tasks/internal", "lib/tasks/dofs",
	"lib/dev/internal", "lib/dev/config", "src/build/ZoneProfiler", "lib/tasks/handle",
	], function (
	immutable, lodash, utils,
	treeOfMatrix, listOfTransform, taskInternal, dofs,
	devInternal, config, ZoneProfiler, handle
	) {

"use strict";

var ln = Math.log,
	exp = Math.exp,
	kSmallestScale = 1e-8;


var kTraceStateChanges = false;

var log = function () {},
	logTreeDiffs = function () {};
if (kTraceStateChanges) {
	log = console.logToUser;

	logTreeDiffs = function (original, updated) {
		var diffs = dofs.Tree.diff(original, updated);

		log("DIFF");
		if (lodash.size(diffs) === 0) {
			log("NONE");
			return;
		}

		lodash.forOwn(diffs, (tuple, key) => {
			log(key + ":");
			var listOriginal = tuple[0],
				listUpdated = tuple[1];

			listOriginal.forEach((dofOriginal, index) => {
				var dofUpdated = listUpdated.get(index);
				if (dofOriginal !== dofUpdated) log(dofOriginal + " -> " + dofUpdated);
			});
		});
	};
}

class Aggregate {
	constructor () {
		this.value = lodash.defaults({ 
				xScale : 0,
				yScale : 0
			}, dofs.TransformDefault); 
		this.mass = lodash.transform(dofs.TransformDefault, function (result, val, key) {
				result[key] = 0;
			}, {});
		this.wasSet = lodash.transform(dofs.TransformDefault, function (result, val, key) {
				result[key] = false;
			}, {});
	}

	add (key, weight, val) {
		if (key === "type") return;

		if (key === "xScale" || key === "yScale") {
			// geometric average
			val = val > kSmallestScale ? val : kSmallestScale;
			this.value[key] += weight * ln(val);
			this.mass[key] += weight;
			this.wasSet[key] = true;
		} else {
			// arithmetic average
			this.value[key] += weight * val;
			this.mass[key] += weight;
			this.wasSet[key] = true;
		}
	}

	average (key, dof) {
		if (key === "type") return dof;

		var val = dof[key],
			type = dof.type;

		var mass = this.mass[key];
		if (key === "xScale" || key === "yScale") {
			// geometric average
			if (mass < 1.0) {
				let lnVal = ln(val);
				val = exp(mass * (this.value[key] - lnVal) + lnVal);
			} else {
				val = exp(this.value[key] / this.mass[key]);			
			}
		} else {
			// arithmetic average
			if (mass < 1.0) {
				val = mass * (this.value[key] - val) + val;
			} else {
				val = this.value[key] / this.mass[key];
			}
		}

		if (mass > 0) {
			type = dofs.type.merge(type, dofs.typeFromKey(key));
		}

		dof.type = type;
		dof[key] = val;
	}
}

var TransformDofs = config.TransformDofs;
devInternal.ifEnabled(TransformDofs, {
	// FIXME: when we have non-instantaneous task remove ongoing tasks according to shouldReset value
	doPreMediateLayer (layer/*, shouldReset*/) {
		layer.tLayerTaskItems = {};
		layer.tLayerPostWarpTaskItems = {};
	},

	doPostMediateLayer: config.MutableTree.enabled ? 
    function (layerPuppet, keyTaskItems) {
    
		var tLayerTaskItems = layerPuppet[keyTaskItems];

		if (lodash.isEmpty(tLayerTaskItems)) return;

		if (kTraceStateChanges) {
			lodash.forOwn(tLayerTaskItems, function (tHandleItems) {
				var layer = tHandleItems.layer;
				console.logToUser("========================================================================");
				console.logToUser("TASKS for Layer '" + layer.getName() + "':");
				lodash.forOwn(tHandleItems, function (items, key) {
					if (key === "layer") return;
					lodash.forEach(items, function (i) {
						console.logToUser("weight: " + i.weight);
						lodash.forOwn(i.task.tDof, function (val, key) {
							console.logToUser(key + ": " + val);
						});
					});
				});
			});
		}

		// aggregate tasks into impending (tomNext) state
		// using next instead of now to account for non-task chanages to the state
		lodash.forOwn(tLayerTaskItems, function (tHandleItems) {
			// retrieve layer and toss that key to leave only handle items
			// FIXME: remove mutation.  instead stash layer next to items --- { layer: ..., items: ...} -- and then iterate over items only
			var layer = tHandleItems.layer;
			delete tHandleItems.layer;

			// derive Transform dofs from impending Matrix dofs
			var	tomPath = taskInternal.getKeyPath(layer, "value");
			var lom = layerPuppet.tomNow.getIn(tomPath.key);

			// update current state with aggregate change...
			var tree = layer.getHandleTreeArray();
			// Aggregate tasks into *sparse* list of transform dofs:
			// entries for handle without tasks will be undefined.
			var lotAverage = [];
            lodash.forOwn(tHandleItems, function (items) {
                // ...get current state
                var handle = items[0].handle,
                    handleRef = tree.getHandleRef(handle),
                    tdof = dofs.transformFromMatrix(lom.getDof(handleRef));

                // ...aggregate tasks
                var aggregate = lodash.transform(items, function (aggregate, item) {
                    var task = item.task,
                        weight = item.weight,
                        tdofNext = task.update(tdof);

                    lodash.forOwn(tdofNext, function (val, key) {
                        aggregate.add(key, weight, val);
                    });
                }, new Aggregate());

                // ...update with average
                var aggregateMass = aggregate.mass;
                lodash.forOwn(aggregateMass, function (mass, key) {
                    if (aggregate.wasSet[key]) {
                        tdof[key] = dofs.TransformDefault[key];
                    }
                    aggregate.average(key, tdof);
                });

                lotAverage[handleRef] = tdof;
            });

			lodash.forEach(lotAverage, function (tdofAvg, index) {
				// ignore undefined entries which correspond to handles without tasks
				if (!tdofAvg) return;

				var typeAvg = tdofAvg.type,
					matAvg = dofs.matFromTransform(tdofAvg);

				var matNow;

				var dofsType = dofs.type;
				lom.setType(index, typeAvg);
                
                switch (typeAvg) {

                    // // ... release without other changes: change type and persist previous state
                    case dofsType.kNotSet:
                        return;

                    // // ... full change: use all of the new dof
                    case dofsType.kAffine: 
                        break;

                    // // ... translation change
                    case dofsType.kTranslation:
                        matNow = lom.getDof(index).matrix;
                        // use linear part from pre-task matrix
                        matAvg[0] = matNow[0];
                        matAvg[1] = matNow[1];
                        matAvg[2] = matNow[2];
                        matAvg[3] = matNow[3];
                        matAvg[4] = matNow[4];
                        matAvg[5] = matNow[5];
                        // keep translation part from task matrix

                        break;

                    // // ... linear change
                    case dofsType.kLinear: 
                        matNow = lom.getDof(index).matrix;
                        // use translation part from pre-task matrix
                        matAvg[6] = matNow[6];
                        matAvg[7] = matNow[7];

                        // keep linear part from task matrix

                        break;
                }
                
                lom.setMatrix(index, matAvg);
			});
		});
	}
    :
    function  (layerPuppet, keyTaskItems) {
		var tLayerTaskItems = layerPuppet[keyTaskItems];

		if (lodash.isEmpty(tLayerTaskItems)) return;

		if (kTraceStateChanges) {
			lodash.forOwn(tLayerTaskItems, function (tHandleItems) {
				var layer = tHandleItems.layer;
				console.logToUser("========================================================================");
				console.logToUser("TASKS for Layer '" + layer.getName() + "':");
				lodash.forOwn(tHandleItems, function (items, key) {
					if (key === "layer") return;
					lodash.forEach(items, function (i) {
						console.logToUser("weight: " + i.weight);
						lodash.forOwn(i.task.tDof, function (val, key) {
							console.logToUser(key + ": " + val);
						});
					});
				});
			});
		}

		// aggregate tasks into impending (tomNext) state
		// using next instead of now to account for non-task chanages to the state
		var mutableTomNext = layerPuppet.tomNext.asMutable();
		lodash.forOwn(tLayerTaskItems, function (tHandleItems) {
			// retrieve layer and toss that key to leave only handle items
			// FIXME: remove mutation.  instead stash layer next to items --- { layer: ..., items: ...} -- and then iterate over items only
			var layer = tHandleItems.layer;
			delete tHandleItems.layer;

			// derive Transform dofs from impending Matrix dofs
			var	tomPath = taskInternal.getKeyPath(layer, "value");
			var lom = mutableTomNext.getIn(tomPath.key);

			// update current state with aggregate change...
			var tree = layer.getHandleTreeArray();
			// Aggregate tasks into *sparse* list of transform dofs:
			// entries for handle without tasks will be undefined.
			var lotAverage = immutable.List().withMutations(function (lot) {
				lodash.forOwn(tHandleItems, function (items) {
					// ...get current state
					var handle = items[0].handle,
						handleRef = tree.getHandleRef(handle),
						tdof = dofs.transformFromMatrix(lom.get(handleRef));

					// ...aggregate tasks
					var aggregate = lodash.transform(items, function (aggregate, item) {
						var task = item.task,
							weight = item.weight,
							tdofNext = task.update(tdof);

						lodash.forOwn(tdofNext, function (val, key) {
							aggregate.add(key, weight, val);
						});
					}, new Aggregate());

					// ...update with average
					var aggregateMass = aggregate.mass;
					lodash.forOwn(aggregateMass, function (mass, key) {
						if (aggregate.wasSet[key]) {
							tdof[key] = dofs.TransformDefault[key];
						}
						aggregate.average(key, tdof);
					});

					lot.set(handleRef, tdof);
				});
			});

			var lomTask = lom.mergeWith(function (mdofNow, tdofAvg) {
				// ignore undefined entries which correspond to handles without tasks
				if (!tdofAvg) return mdofNow;

				var typeAvg = tdofAvg.type,
					matAvg = dofs.matFromTransform(tdofAvg);

				var matNow;

				var dofsType = dofs.type;
				return mdofNow.withMutations(function (mdofNow) {
					mdofNow.set("type", typeAvg);

					switch (typeAvg) {

						// // ... release without other changes: change type and persist previous state
						case dofsType.kNotSet:
							return mdofNow;

						// // ... full change: use all of the new dof
						case dofsType.kAffine: 
							return mdofNow.set("matrix", matAvg);

						// // ... translation change
						case dofsType.kTranslation: 
							matNow = mdofNow.get("matrix");
							// use linear part from pre-task matrix
							matAvg[0] = matNow[0];
							matAvg[1] = matNow[1];
							matAvg[2] = matNow[2];
							matAvg[3] = matNow[3];
							matAvg[4] = matNow[4];
							matAvg[5] = matNow[5];
							// keep translation part from task matrix

							return mdofNow.set("matrix", matAvg);

						// // ... linear change
						case dofsType.kLinear: 
							matNow = mdofNow.get("matrix");
							// use translation part from pre-task matrix
							matAvg[6] = matNow[6];
							matAvg[7] = matNow[7];

							// keep linear part from task matrix

							return mdofNow.set("matrix", matAvg);
					}
				});
			}, lotAverage);

			mutableTomNext.setIn(tomPath.key, lomTask);
		});

		layerPuppet.tomNext = mutableTomNext.asImmutable();
	},

	log (format) {
		devInternal.printf("TransformDofs: " + format, arguments, 1);
	}
});

/*============================================================================
	Matrix Dofs
============================================================================*/


var dofsSetOnChange = dofs.setOnChange,
	dofsUpdateOnChange = dofs.updateOnChange,
	dofsUpdateOnMatrixChange = dofs.updateOnMatrixChange;

function updateOnMatrixChange (lomNow, lomNext) {
	var lomMerge = lomNow.withMutations(function (lomNow) {
		lomNow.forEach(function (valNow, key) {
			var valNext = lomNext.get(key);
			var val = dofsUpdateOnMatrixChange(valNow, valNext);
			return dofsSetOnChange(lomNow, key, valNow, val);
		});
	});

	return lomMerge;
}


/*
 * Unaltered dofs are released automatically so that warper can return them to rest.
 * If a layer contains only released dofs then all of its dofs are returned to rest immediately.
 * NOTE: This function can become a hotspot so the loop combines release and counting of released handles.
 */
function releaseUntouchedHandlesAndReturnEntireMeshIfAllReleased (lomNow, lomBirth) {
	var nReleased = 0;

    lomNow.aDof.forEach(function (valNow, i) {
        if (!lomNow.hasDofBeenTouchedSinceLastIncrement(i)) {
            // release untouched dof...
            nReleased += 1;
        	lomNow.setType(i, lomBirth.getType(i));

        } else if (lomNow.getType(i) === lomBirth.getType(i)) {
            // was previously released or never touched
            nReleased += 1;
        }
    });

    // return to rest immediately when all handles are released
    if (lomNow.getNumDofs() === nReleased) {
        lomBirth.aDof.forEach(function (valBirth, i) {
            lomNow.setDof(i, lodash.cloneDeep(valBirth));
        });
    }
}

/*
 * Merges incoming dofs (lomNext) with the current dofs (lomNow).
 * Unaltered dofs are released automatically so that warper can return them to rest.
 * If a layer contains only released dofs then all of its dofs are returned to rest immediately.
 * NOTE: This function can become a hotspot so the loop combines release and counting of released handles.
 */
function updateOnMatrixChangeAndReturn (lomNow, lomNext, lomBirth) {
	var nReleased = 0;
	var lomMerge = lomNow.withMutations(function (lomNow) {
		lomNow.forEach(function (valNow, key) {
			var valNext = lomNext.get(key),
				valBirth = lomBirth.get(key),
				typeBirth = valBirth.get("type");

			// release untouched dof...
			if (valNow === valNext) {
				nReleased += 1;
				var updated = dofsUpdateOnChange(valNow, "type", () => typeBirth);
				if (updated !== valNow) {
					log("=======");
					log("RELEASE");
					log("=======");
					return lomNow.set(key, updated);
				}

				return lomNow;
			}

			// ...or merge incoming change
			var val = dofsUpdateOnMatrixChange(valNow, valNext);
			if (val.get("type") === typeBirth) nReleased += 1;
			return dofsSetOnChange(lomNow, key, valNow, val);
		});
	});

	// return to rest immediately when all handles are released
	if (lomNow.size === nReleased) {
		return lomBirth;
	} else {
		return lomMerge;
	}
}
    
    
var commitMoveFrameBy = config.MutableTree.enabled ?
    function (layer) {
        const accumulator = handle.internal.getMoveFrameByAccumulator(layer.getSdkLayer());

        lodash.forOwn(accumulator, function (moveByItem) {
            var keyArray = moveByItem.keyArray,
                matMove = moveByItem.value;

/* FIXME: currently moveFrameBy overrides any setFrame, but the following doesn't stop movement across frames
            tomNext.updateIn(keyArray, function (mdof) {
                return new dofs.Matrix({
                    type: dofs.type.kAffine,
                    matrix : mat3.multiply(mdof.get("matrix"), matMove)
                });
            });
*/

            layer.tomNow.setIn(keyArray, new dofs.Matrix({
                    type: dofs.type.kAffine,
                    matrix : matMove
            }));
        });
    }
:
    function (layer) {
        const accumulator = handle.internal.getMoveFrameByAccumulator(layer.getSdkLayer());

        return layer.tomNext.withMutations(function (tomNext) {
            lodash.forOwn(accumulator, function (moveByItem) {
                var keyArray = moveByItem.keyArray,
                    matMove = moveByItem.value;

    /* FIXME: currently moveFrameBy overrides any setFrame, but the following doesn't stop movement across frames
                tomNext.updateIn(keyArray, function (mdof) {
                    return new dofs.Matrix({
                        type: dofs.type.kAffine,
                        matrix : mat3.multiply(mdof.get("matrix"), matMove)
                    });
                });
    */

                tomNext.setIn(keyArray, new dofs.Matrix({
                        type: dofs.type.kAffine,
                        matrix : matMove
                }));
            });
        });
    };


var MatrixDofs = config.MatrixDofs,
	treeUpdateValuesWithZipped = dofs.Tree.updateValuesWithZipped;

devInternal.ifEnabled(MatrixDofs, config.MutableTree.enabled ? {
    
	doPreMediateLayer (layer, shouldReset) {
		utils.assert(layer.tomNow, "Layer state not found.");
        
		if (shouldReset) {
			// begin with clone of initial state so it can be mutated
			layer.tomNow = layer.tomBirth.clone();
            layer.tomPrev = layer.tomBirth;     // tomPrev and tomBirth are both read-only, so we don't need to clone
            layer.tomPrevPrev = layer.tomPrev;
		}
	},

	doPostMediateLayer: function (layer) {
		var tomPrev = layer.tomPrev,
			tomNow = layer.tomNow,
			tomBirth = layer.tomBirth;

		commitMoveFrameBy(layer);

		ZoneProfiler.push("releaseUntouched");
		treeUpdateValuesWithZipped(releaseUntouchedHandlesAndReturnEntireMeshIfAllReleased, tomNow, tomBirth);
		ZoneProfiler.pop("releaseUntouched");

		// log(`NEXT\n${tomNext}\nWITHRETURN\n${tomNextWithReturn}`);
		// if (tomNow === tomNextWithReturn) {
		// 	log(`tomNow === tomNextWithReturn: ${tomNow === tomNextWithReturn}`);
		// }

		ZoneProfiler.push("Warp");
		treeOfMatrix.warp(layer, layer.tomPrevPrev, tomPrev, tomNow);
		ZoneProfiler.pop("Warp");

		layer.tomPrevPrev = layer.tomPrev;
        layer.tomPrev = layer.tomNow.clone();       // TODO: remove this once we get back boolean from warp() C++ code to indicate that it has converged,
                                                    //  then we can stop comparing against this state; also: factor with commitMatrixDofs
	},

	updateOnMatrixChange,

	log (format) {
		devInternal.printf("MatrixDofs: " + format, arguments, 1);
	},

	logTreeDiffs
} : {   // To be removed
	doPreMediateLayer (layer, shouldReset) {
		// now - represents the currently drawn state
		utils.assert(layer.tomNow, "Layer state not found.");

		// next - represents the state that will be drawn next
		var tomNext;
		if (shouldReset) {
			// begin with initial state...
			tomNext = layer.tomBirth;
		} else {
			// ... or continue from where we were in previous frame
			tomNext = layer.tomNow;
		}

		layer.tomNext = tomNext;
	},

	doPostMediateLayer: function (layer) {
		var tomPrev0 = layer.tomPrev,
			tomNow = layer.tomNow,
			tomNext = layer.tomNext,
			tomBirth = layer.tomBirth;

		if (kTraceStateChanges) {
			log("==========================================================================");
			log(`BASE layer '${layer.getName()}', puppet '${layer.getPuppet().getName()}'\n\tnow\n${tomNow}\n\tnext\n${tomNext}`);

			log(`DIFF now vs next:`);
			logTreeDiffs(tomNow, tomNext);
		}

		tomNext = commitMoveFrameBy(layer);

		ZoneProfiler.push("MergeWithReturn");
		var tomNextWithReturn = treeUpdateValuesWithZipped(updateOnMatrixChangeAndReturn, tomNow, tomNext, tomBirth);
		ZoneProfiler.pop("MergeWithReturn");

		// log(`NEXT\n${tomNext}\nWITHRETURN\n${tomNextWithReturn}`);
		// if (tomNow === tomNextWithReturn) {
		// 	log(`tomNow === tomNextWithReturn: ${tomNow === tomNextWithReturn}`);
		// }

		ZoneProfiler.push("Warp");
		var tomWarp = treeOfMatrix.warp(layer, tomPrev0, tomNow, tomNextWithReturn);
		ZoneProfiler.pop("Warp");

		if (kTraceStateChanges && tomWarp !== tomNow) {
			log(`DIFF now vs warp`);
			logTreeDiffs(tomNow, tomWarp);
		}

		layer.tomPrev = tomNow;
		layer.tomNow = tomWarp;
		layer.tomNext = tomWarp;
	},

	updateOnMatrixChange,

	log (format) {
		devInternal.printf("MatrixDofs: " + format, arguments, 1);
	},

	logTreeDiffs
});


var Debug = config.Debug;
return {
	Debug,
	MatrixDofs,
	TransformDofs
};


}); // end define